/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

package server

import (
	"aws/amazon-gamelift-go-sdk/model"
)

// ServerParameters - object communicating the following information about the server
// communication with GameLift from an Anywhere server. This parameter isn't required for servers hosted on
// GameLift managed EC2 instances.
//
//   - WebSocketURL - the URL of the websocket used to connect to your game server.
//   - ProcessID - the ID of the process used to host your game server.
//   - HostID - the ID of the compute hosting your game server processes.
//   - FleetID - the ID of the GameLift fleet containing your Anywhere compute.
//   - AuthToken - the authorization token generated by the GameLift operation.
//   - AwsRegion - the AWS Region of the Game Server.
//   - AccessKey - the AWS AccessKey of the AWS Credentials with GameLift Access.
//   - SecretKey - the AWS SecretKey of the AWS Credentials with GameLift Access.
//   - SessionToken - the AWS Token of the AWS Credentials with GameLift Access if using temporary credentials.
type ServerParameters struct {
	WebSocketURL string
	ProcessID    string
	HostID       string
	FleetID      string
	AuthToken    string
	AwsRegion    string
	AccessKey    string
	SecretKey    string
	SessionToken string
}

// ProcessParameters - object that communicating the following information about the server process:
//
//   - Names of callback methods, implemented in the game server code,
//     that the GameLift service invokes to communicate with the server process.
//   - Port number that the server process is listening on.
//   - Path to any game session-specific files that you want GameLift to capture and store.
type ProcessParameters struct {
	// OnStartGameSession - callback function that the GameLift service invokes to activate a new game session.
	// GameLift calls this function in response to the client request CreateGameSession.
	// The callback function passes a model.GameSession object.
	OnStartGameSession func(model.GameSession)

	// OnUpdateGameSession - callback function that the GameLift service invokes to pass
	// an updated game session object to the server process.
	// GameLift calls this function when a match backfill request has been processed
	// in order to provide updated matchmaker data.
	// It passes a GameSession object, a status update (updateReason), and the match backfill ticket ID.
	OnUpdateGameSession func(model.UpdateGameSession)

	// OnProcessTerminate - callback function that the GameLift service invokes to force
	// the server process to shut down.
	// After calling this function, GameLift waits five minutes for the server process to shut down
	// and respond with a ProcessEnding() call.
	// If no response is received, it shuts down the server process.
	OnProcessTerminate func()

	// OnHealthCheck - callback function that the GameLift service invokes to request a health status report
	// from the server process.
	// GameLift calls this function every 60 seconds.
	// After calling this function GameLift waits 60 seconds for a response,
	// and if none is received. records the server process as unhealthy.
	OnHealthCheck func() bool

	// Port - the server process listens on for new player connections.
	// The value must fall into the port range configured for any fleet deploying this game server build.
	// This port number is included in game session and player session objects,
	// which game sessions use when connecting to a server process.
	Port int

	// Object with a list of directory paths to game session log files.
	LogParameters LogParameters
}

// LogParameters - this data type is used to identify which files generated during a game session
// that you want GameLift to upload and store once the game session ends.
type LogParameters struct {
	// Directory paths to game server log files that you want GameLift to store for future access.
	// These files are generated during each game session.
	// File paths and names are defined in your game server and stored in the root game build directory.
	// The log paths must be absolute.
	// For example, if your game build stores game session logs in a path like MyGame\sessionlogs\,
	// then the log path would be c:\game\MyGame\sessionLogs (on a Windows instance)
	// or /local/game/MyGame/sessionLogs (on a Linux instance).
	LogPaths []string
}
