/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

package request

import (
	"aws/amazon-gamelift-go-sdk/model/message"
)

// ActivateServerProcessRequest - This request is sent to the GameLift WebSocket during the ProcessReady call.
//
// Please use NewActivateServerProcess function to create this request.
type ActivateServerProcessRequest struct {
	message.Message
	// The value includes the SDK version number.
	SdkVersion string `json:"SdkVersion,omitempty"`
	// The value includes the SDK language.
	SdkLanguage string `json:"SdkLanguage,omitempty"`
	// The value includes the SDK tool name (if applicable).
	SdkToolName string `json:"SdkToolName,omitempty"`
	// The value includes the SDK tool version number (if applicable)
	SdkToolVersion string `json:"SdkToolVersion,omitempty"`
	// Port number the server process listens on for new player connections.
	// The value must fall into the port range configured for any fleet deploying this game server build.
	// This port number is included in game session and player session objects,
	// which game sessions use when connecting to a server process.
	Port int `json:"Port,omitempty"`
	// LogPaths - List of directory paths to game server log files you want GameLift
	// to store for future access. These files are generated by a server
	// process during each game session; file paths and names are defined
	// in your game server and stored in the root game build directory.
	// The log paths must be absolute.
	LogPaths []string `json:"LogPaths"`
}

// NewActivateServerProcess - creates a new ActivateServerProcessRequest
// generates a RequestID to match the request and response.
func NewActivateServerProcess(sdkVersion, sdkLanguage string, port int) ActivateServerProcessRequest {
	return ActivateServerProcessRequest{
		Message:     message.NewMessage(message.ActivateServerProcess),
		SdkVersion:  sdkVersion,
		SdkLanguage: sdkLanguage,
		Port:        port,
	}
}
