/*
* All or portions of this file Copyright (c) Amazon.com, Inc. or its affiliates or
* its licensors.
*
* For complete copyright and license terms please see the LICENSE at the root of this
* distribution (the "License"). All use of this software is governed by the License,
* or, if provided, by the license below or the license accompanying this file. Do not
* remove or modify any license notices. This file is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*
*/

using System;
using Aws.GameLift.Server.Model;

namespace Aws.GameLift.Server
{
    public static class GameLiftServerAPI
    {
        private static readonly string SDK_VERSION = "5.0.0";

        /// <summary>
        /// Returns the current version number of the SDK built into the server process.
        /// </summary>
        /// <returns>If successful, returns the current SDK version as an AwsStringOutcome object.
        /// The returned string includes the version number only (ex. <c>5.0.0</c>). If not successful,
        /// returns an error message.</returns>
        /// <example>
        /// <code>var getSdkVersionOutcome = GameLiftServerAPI.GetSdkVersion();</code>
        /// </example>
        public static AwsStringOutcome GetSdkVersion()
        {
            return new AwsStringOutcome(SDK_VERSION);
        }

        /// <summary>
        /// Initializes the GameLift server SDK. This method should be called on launch, before any other
        /// GameLift-related initialization occurs.
        /// </summary>
        /// <param name="serverParameters">
        /// A <see cref="ServerParameters"/> object communicating the following information about the server 
        /// communication with GameLift from an Anywhere server. This parameter isn't required for servers hosted on 
        /// GameLift managed EC2 instances.
        /// <list type="bullet">
        /// <item>The URL of the websocket used to connect to your game server.</item>
        /// <item>The ID of the process used to host your game server.</item>
        /// <item>The ID of the compute hosting your game server processes.</item>
        /// <item>The ID of the GameLift fleet containg your Anywhere compute</item>
        /// <item>The authorization token generated by the GameLift operation
        /// <a href="https://docs.aws.amazon.com//gamelift/latest/apireference/API_GetComputeAuthToken.html">
        /// GetComputeAuthToken</a>. Your game server automatically maintains the 
        /// connection, until you close the server.
        /// </item>
        /// </list>
        /// </param>
        /// <returns>If successful, returns an InitSdkOutcome object indicating that the server process is
        /// ready to call <see cref="ProcessReady"/>
        /// </returns>
        /// <example>
        /// <code>
        /// ServerParameters serverParameters = new ServerParameters(
        ///     webSocketUrl,
        ///     processId,
        ///     hostId,
        ///     fleetId);
        /// 
        /// // InitSDK will establish a local connection with GameLift's agent to enable further communication.
        /// var initSDKOutcome = GameLiftServerAPI.InitSDK(serverParameters);
        /// </code>
        /// </example>
        public static GenericOutcome InitSDK(ServerParameters serverParameters)
        {
            return ServerState.Instance.InitializeNetworking(serverParameters);
        }

        /// <summary>
        /// Notifies GameLift that the server process is ready to host game sessions. Call this
        /// method after successfully invoking <see cref="InitSDK"/> and completing setup tasks that are required
        /// before the server process can host a game session. This method should be called only once per process.
        /// </summary>
        /// <param name="processParameters">
        /// An <see cref="ProcessParameters"/> object communicating the following information about the server process:<br/>
        /// <list type="bullet">
        /// <item>Names of callback methods, implemented in the game server code, that GameLift invokes to
        /// communicate with the server process.</item>
        /// <item>Port number that the server process is listening on.</item>
        /// <item>Path to any game session-specific files that you want GameLift to capture and store.</item>
        /// </list>
        /// </param>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// // Set parameters and call ProcessReady()
        /// var processParams = new ProcessParameters(
        ///     this.OnGameSession,
        ///     this.OnProcessTerminate,
        ///     this.OnHealthCheck,
        ///     this.OnGameSessionUpdate,
        ///     port,
        ///     new LogParameters(new list &lt; string &gt; (){   // logging and error example
        ///         "C:\\game\\logs",
        ///         "C:\\game\\error"
        ///         })
        /// );
        ///     
        /// var processReadyOutcome = GameLiftServerAPI.ProcessReady(processParams);
        /// </code>
        /// </example>
        public static GenericOutcome ProcessReady(ProcessParameters processParameters)
        {            
            return ServerState.Instance.ProcessReady(processParameters);
        }

        /// <summary>
        /// Notifies GameLift that the server process is shutting down. This method should be called after
        /// all other cleanup tasks, including shutting down all active game sessions. This method should exit with an
        /// exit code of 0; a non-zero exit code results in an event message that the process did not exit cleanly.
        /// <br/>
        /// After the method exits with a code of 0, you can terminate the process with a successful exit code.
        /// You can also exit the process with an error code. If you exit with an error code, the fleet event
        /// will indicated the process terminated abnormally (<c>SERVER_PROCESS_TERMINATED_UNHEALTHY</c>).
        /// </summary>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// var processEndingOutcome = GameLiftServerAPI.ProcessEnding();
        /// if (processReadyOutcome.Success)
        ///     Environment.Exit(0);
        /// // otherwise, exit with error code
        /// Environment.Exit(errorCode);
        /// </code>
        /// </example>
        public static GenericOutcome ProcessEnding()
        {
            return ServerState.Instance.ProcessEnding();
        }

        /// <summary>
        /// Notifies GameLift that the server is requesting a game session and is now ready to
        /// receive player connections. This action should be called as part of the <c>onStartGameSession()</c> callback
        /// function, after all game session initialization has been completed.
        /// </summary>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// void OnStartGameSession(GameSession gameSession){
        ///     // game-specific tasks when starting a new game session, such as loading map
        ///     
        ///     // When ready to receive players
        ///     var activateGameSessionOutcome = GameLiftServerAPI.ActivateGameSession();
        /// </code>
        /// </example>
        public static GenericOutcome ActivateGameSession()
        {
            return ServerState.Instance.ActivateGameSession();
        }

        /// <summary>
        /// Updates the current game session's ability to accept new player sessions.
        /// A game session can be set to either accept or deny all new player sessions.
        /// <seealso cref="UpdateGameSession"/>
        /// </summary>
        /// <param name="playerSessionPolicy">String value indicating whether the game session accepts new players.
        /// Valid values include:
        /// <list type="bullet">
        /// <item> ACCEPT_ALL – Accept all new player sessions. </item>
        /// <item> DENY_ALL – Deny all new player sessions. </item>
        /// </list>
        /// </param>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// var updatePlayerSessionCreationPolicyOutcomex = GameLiftServerAPI.UpdatePlayerSessionCreationPolicy(PlayerSessionCreationPolicy.ACCEPT_ALL);
        /// </code>
        /// </example>
        public static GenericOutcome UpdatePlayerSessionCreationPolicy(PlayerSessionCreationPolicy playerSessionPolicy)
        {
            return ServerState.Instance.UpdatePlayerSessionCreationPolicy(playerSessionPolicy);
        }

        /// <summary>
        /// Retrieves the ID of the game session currently being hosted by the server process,
        /// if the server process is active.
        /// <br/>
        /// For idle process that are not yet activated with a game session, the call returns
        /// <c>Success=True</c> and <c>GameSessionId=""</c> (an empty string).
        /// </summary>
        /// <returns>If successful, returns the game session ID as an <see cref="AwsStringOutcome"/> object.
        /// If not successful, returns an error message.</returns>
        /// <example>
        /// <code>var getGameSessionIdOutcome = GameLiftServerAPI.GetGameSessionId();</code>
        /// </example>
        public static AwsStringOutcome GetGameSessionId()
        {
            return ServerState.Instance.GetGameSessionId();
        }

        /// <summary>
        /// Returns the time that a server process is scheduled to be shut down, if a termination time is available.
        /// A server process takes this action after receiving an <c>onProcessTerminate()</c> callback from GameLift. 
        /// GameLift may call <c>onProcessTerminate()</c> for the following reasons:
        /// <list type="number">
        /// <item>For poor health (the server process has reported port health or has not responded to GameLift</item>
        /// <item>When terminating the instance during a scale-down event</item>
        /// <item>when an instance is being terminated due to a
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/spot-tasks.html">spot-instance interruption</a></item>
        /// </list>
        /// <br/>
        /// If the process has received an <c>onProcessTerminate()</c> callback, the value returned is the estimated
        /// termination time. If the process has not received an <c>onProcessTerminate()</c> callback,
        /// an error message is returned. Learn more about
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-terminate">
            /// shutting down a server process</a>.
        /// </summary>
        /// <returns>If successful, returns the termination time as an AwsDateTimeOutcome object.
        /// The value is the termination time, expressed in elapsed ticks since 0001 00:00:00. For example,
        /// the date time value 2020-09-13 12:26:40 -000Z is equal to 637355968000000000 ticks.
        /// If no termination time is available, returns an error message.</returns>
        /// <example><c>var getTerminationTimeOutcome = GameLiftServerAPI.GetTerminationTime();</c></example>
        public static AwsDateTimeOutcome GetTerminationTime()
        {
            return ServerState.Instance.GetTerminationTime();
        }

        /// <summary>
        /// Notifies the GameLift service that a player with the specified player session ID has connected to the
        /// server process and needs validation. GameLift verifies that the player session ID 
        /// has reserved a player slot in the game session. Once validated,
        /// GameLift changes the status of the player slot from <c>RESERVED</c> to <c>ACTIVE</c>.
        /// </summary>
        /// <param name="playerSessionId">Unique ID issued by GameLift when a new player session is created.</param>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// void ReceiveConnectingPlayerSessionID (Connection connection, String playerSessionId){
        ///     var acceptPlayerSessionOutcome = GameLiftServerAPI.AcceptPlayerSession(playerSessionId);
        ///     if(acceptPlayerSessionOutcome.Success){
        ///         connectionToSessionMap.emplace(connection, playerSessionId);
        ///         connection.Accept();
        ///     }
        ///     else{
        ///         connection.Reject(acceptPlayerSessionOutcome.Error.ErrorMessage); 
        ///     }       
        /// }
        /// </code>
        /// </example>
        public static GenericOutcome AcceptPlayerSession(String playerSessionId)
        {
            return ServerState.Instance.AcceptPlayerSession(playerSessionId);
        }

        /// <summary>
        /// Notifies the GameLift service that a player with the specified player session ID has disconnected from the
        /// server process. In response, GameLift changes the player slot to available, which allows it to be
        /// assigned to a new player.
        /// </summary>
        /// <param name="playerSessionId">Unique ID issued by GameLift when a new player session is created.</param>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// Aws::GameLift::GenericOutcome disconnectOutcome = Aws::GameLift::Server::RemovePlayerSession(playerSessionId);
        /// </code>
        /// </example>
        public static GenericOutcome RemovePlayerSession(String playerSessionId)
        {
            return ServerState.Instance.RemovePlayerSession(playerSessionId);
        }

        /// <summary>
        /// Retrieves player session data, including settings, session metadata, and player data.
        /// Use this action to get information for a single player session, for all player sessions in a game session,
        /// or for all player sessions associated with a single player ID.
        /// </summary>
        /// <param name="request">A <see cref="DescribePlayerSessionsRequest"/> object describing which player sessions
        /// to retrieve.</param>
        /// <returns>If successful, returns a DescribePlayerSessionsOutcome object containing a set of player session
        /// objects that fit the request parameters.</returns>
        /// <example>
        /// This example illustrates a request for all player sessions actively connected to a specified game session. 
        /// By omitting <c>NextToken</c> and setting the <c>Limit</c> value to 10, GameLift will return the first 10 player sessions 
        /// records matching the request.
        /// <code>
        /// // Set request parameters 
        /// var describePlayerSessionsRequest = new Aws.GameLift.Server.Model.DescribePlayerSessionsRequest(){
        ///     GameSessionId = GameLiftServerAPI.GetGameSessionId().Result,    //gets the ID for the current game session
        ///     Limit = 10,
        ///     PlayerSessionStatusFilter = PlayerSessionStatusMapper.GetNameForPlayerSessionStatus(PlayerSessionStatus.ACTIVE)
        /// };
        /// 
        /// // Call DescribePlayerSessions
        /// Aws::GameLift::DescribePlayerSessionsOutcome playerSessionsOutcome = Aws::GameLift::Server::Model::DescribePlayerSessions(describePlayerSessionRequest);
        /// </code>
        /// </example>
        public static DescribePlayerSessionsOutcome DescribePlayerSessions(DescribePlayerSessionsRequest request)
        {
            return ServerState.Instance.DescribePlayerSessions(request);
        }

        /// <summary>Sends a request to find new players for open slots in a game session created with FlexMatch.
        /// See the AWS SDK action <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html">StartMatchBackfill()</a>. 
        /// With this action, match backfill requests can be
        /// initiated by a game server process that is hosting the game session. Learn more about the
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">FlexMatch backfill
        /// feature</a>.
        /// <br/>
        /// This action is asynchronous. If new players are successfully matched, GameLift delivers updated
        /// matchmaker data using the callback function OnUpdateGameSession().
        /// <br/>
        /// A server process can have only one active match backfill request at a time.
        /// To send a new request, first call <see cref="StopMatchBackfill"/> to cancel the original request.
        /// </summary>
        /// <param name="request">A <see cref="StartMatchBackfillRequest"/> object that communicates the following
        /// information:
        /// <list type="bullet">
        /// <item>A ticket ID to assign to the backfill request. This information is optional; if no ID is provided,
        /// GameLift will autogenerate one. </item>
        /// <item>The matchmaker to send the request to. The full configuration ARN is required.
        /// This value can be acquired from the game session's matchmaker data.</item>
        /// <item>The ID of the game session that is being backfilled.</item>
        /// <item>Available matchmaking data for the game session's current players.</item>
        /// </list>
        /// </param>
        /// <returns>If successful, returns a StartMatchBackfillOutcome object with the match backfill 
        /// ticket id or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// // Build a backfill request
        /// var startBackfillRequest = new AWS.GameLift.Server.Model.StartMatchBackfillRequest(){
        ///     TicketId = "a ticket ID",   // optional
        ///     MatchmakingConfigurationArn = "the matchmaker configuration ARN",
        ///     GameSessionId = GameLiftServerAPI.GetGameSessionId().Result,    // gets ID for current game session
        ///     //get player data for all currently connected players
        ///     MatchmakerData matchmakerData = MatchmakerData.FromJson(gameSession.MatchmakerData);  // gets matchmaker data for current players
        ///     // get matchmakerData.Players
        ///     // remove data for players who are no longer connected
        ///     Players = ListOfPlayersRemainingInTheGame
        /// };
        /// // Send backfill request
        /// var startBackfillOutcome = GameLiftServerAPI.StartMatchBackfill(startBackfillRequest);
        /// // Implement callback function for backfill
        /// void OnUpdateGameSession(GameSession myGameSession){
        /// // game-specific tasks to prepare for the newly matched players and update matchmaker data as needed  
        /// }
        /// </code>
        /// </example>
        public static StartMatchBackfillOutcome StartMatchBackfill(StartMatchBackfillRequest request)
        {
            return ServerState.Instance.StartMatchBackfill(request);
        }

        /// <summary>
        /// Cancels an active match backfill request that was created with StartMatchBackfill(). Learn more about the
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">FlexMatch backfill
        /// feature</a>.
        /// </summary>
        /// <param name="request">A <see cref="StopMatchBackfillRequest"/> object identifying the matchmaking ticket to cancel:
        /// <list type="bullet">
        /// <item>Ticket ID assigned to the backfill request being canceled</item>
        /// <item>Matchmaker the backfill request was sent to</item>
        /// <item>Game session associated with the backfill request</item>
        /// </list>
        /// </param>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message.</returns>
        /// <example>
        /// <code>
        /// // Set backfill stop request parameters
        /// var stopBackfillRequest = new AWS.GameLift.Server.Model.StopMatchBackfillRequest(){
        ///     TicketId = "a ticket ID", //optional, if not provided one is autogenerated
        ///     MatchmakingConfigurationArn = "the matchmaker configuration ARN", //from the game session matchmaker data
        ///     GameSessionId = GameLiftServerAPI.GetGameSessionId().Result    //gets the ID for the current game session
        /// };
        ///
        /// var stopBackfillOutcome = GameLiftServerAPI.StopMatchBackfillRequest(stopBackfillRequest);
        /// </code>
        /// </example>
        public static GenericOutcome StopMatchBackfill(StopMatchBackfillRequest request)
        {
            return ServerState.Instance.StopMatchBackfill(request);
        }

        /// <summary>
        /// Retreives the path to TLS certificate used to encrypt the network connection between your
        /// Anywhere compute resource and GameLift. You can use the certificate path when you register
        /// your compute device to a GameLift Anywhere fleet. For more information see, 
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_RegisterCompute.html">RegisterCompute</a>.
        /// </summary>
        /// <returns>Returns a <see cref="GetComputeCertificateResponse"/> object containing the following
        /// <list type="bullet">
        /// <item>CertificatePath - The path to the TLS certificate on your compute resource.</item>
        /// <item>HostName - The hostname of your compute resource.</item>
        /// </list>
        /// </returns>
        /// <example>
        /// <code>
        /// var GetComputeCertificateOutcome TlsCertificate = GameLiftServerAPI.GetComputeCertificate();
        /// </code>
        /// </example>
        public static GetComputeCertificateOutcome GetComputeCertificate()
        {
            return ServerState.Instance.GetComputeCertificate();
        }

        /// <summary>
        /// Retrieves the service role credentials you created to extend permissions to
        /// your other AWS services to GameLift. These credentials allow your game server to 
        /// use your AWS resources. For more information, see 
        /// <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">
        /// Setting up an IAM service role</a>.
        /// </summary>
        /// <returns>
        /// Returns a <see cref="GetFleetRoleCredentialsOutcome"/> object that contains the following:
        /// <list type="bullet">
        /// <item>AssumedRoleUserArn - The Amazon Resource Name (ARN) of the user that the service role belongs to.</item>
        /// <item>AssumedRoleId - The ID of the user that the service role belongs to.</item>
        /// <item>AccessKeyId - The access key ID used to authenticate and provide access to your AWS resources.</item>
        /// <item>SecretAccessKey - The secret access key ID used for authentication.</item>
        /// <item>SessionToken - A token used to identify the current active session interacting with your AWS resources.</item>
        /// <item>Expiration - The amount of time until your session credentials expire.</item>
        /// </list>
        /// <paramref name="request">A <see cref="GetFleetRoleCredentialsRequest"/> object identifying the Amazon 
        /// Resource Name of the role you are requesting the credentials of.</paramref>
        /// </returns>
        /// <example>
        /// <code>
        /// // form the customer credentials request
        /// var GetFleetRoleCredentialsRequest = new AWS.GameLift.Server.Model.GetFleetRoleCredentialsRequest(){
        ///     RoleArn = "arn:aws:iam::123456789012:role/service-role/exampleGameLiftAction"
        ///  }
        ///  
        /// var GetFleetRoleCredentialsOutcome credentials = GetFleetRoleCredentials(GetFleetRoleCredentialsRequest)
        /// </code>
        /// </example>
        public static GetFleetRoleCredentialsOutcome GetFleetRoleCredentials(GetFleetRoleCredentialsRequest request)
        {
            return ServerState.Instance.GetFleetRoleCredentials(request);
        }

        /// <summary>
        /// Deletes the instance of the GameLift Game Server SDK on your resource. This removes all state 
        /// information, stops heartbeat communication with GameLift, stops game session management, and
        /// closes any connections. Call this after you've use <see cref="ProcessEnding"/>
        /// </summary>
        /// <returns>Returns a generic outcome consisting of success or failure with an error message. </returns>
        /// <example>
        /// <code>
        /// // operations to end game sessions and the server process
        /// var processEndingOutcome = GameLiftServerAPI.ProcessEnding();
        /// if (processReadyOutcome.Success)
        ///     Environment.Exit(0);
        /// // otherwise, exit with error code
        /// Environment.Exit(errorCode);
        /// 
        /// // Shutdown and destroy the intance of the GameLift Game Server SDK
        /// var destroyOutcome = GameLiftServerAPI.Destroy();
        /// </code>
        /// </example>
        public static GenericOutcome Destroy()
        {
            ServerState.Instance.Shutdown();
            return new GenericOutcome();
        }
    }
}